<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    theme_enlight
 * @copyright  2015 onwards LMSACE Dev Team (http://www.lmsace.com)
 * @authors    LMSACE Dev Team
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

/**
 * Enlight main scss contents.
 * @param type $theme
 * @return type
 */
function theme_enlight_get_main_scss_content($theme) {
    global $CFG;

    $scss = '';
    $parentconfig = theme_config::load('boost');
    $scss .= theme_boost_get_main_scss_content($parentconfig);

    $themeScssfile = $CFG->dirroot.'/theme/enlight/scss/preset/enlight.scss';
    if ( file_exists($themeScssfile) ) {
        $scss .= file_get_contents($themeScssfile);
    }

    $patternNum = theme_enlight_get_setting('patternselect');

    if ( !empty($patternNum) ) {
        $filename = 'pattern-'.$patternNum.'.scss';
    } else {
        $filename = 'pattern-default.scss';
    }

    $fs = get_file_storage();

    $context = context_system::instance();
    if ($filename == 'pattern-1.scss') {
        $scss .= file_get_contents($CFG->dirroot . '/theme/enlight/scss/preset/pattern-1.scss');

    } else if ($filename == 'pattern-2.scss') {
        $scss .= file_get_contents($CFG->dirroot . '/theme/enlight/scss/preset/pattern-2.scss');

    } else if ($filename == 'pattern-3.scss') {
        $scss .= file_get_contents($CFG->dirroot . '/theme/enlight/scss/preset/pattern-3.scss');

    } else if ($filename == 'pattern-4.scss') {
        $scss .= file_get_contents($CFG->dirroot . '/theme/enlight/scss/preset/pattern-4.scss');

    } else if ($filename && ($presetfile = $fs->get_file($context->id, 'theme_enlight', 'preset', 0, '/', $filename))) {
        $scss .= $presetfile->get_content();

    } else {
        // Safety fallback - maybe new installs etc.
        $scss .= file_get_contents($CFG->dirroot . '/theme/enlight/scss/preset/pattern-default.scss');
    }

    return $scss;
}

/**
 * override the scss values with variables.
 * @param type $theme
 * @return type
 */
function theme_enlight_get_pre_scss($theme) {
    global $CFG;

    $scss = '';
    $bgImgs = theme_enlight_load_bgimages($theme, $scss);
    if (!empty($bgImgs)) {
        $scss .= $bgImgs;
    }

     // Prepend pre-scss.
    if (!empty($theme->settings->scsspre)) {
        $scss .= $theme->settings->scsspre;
    }

    $scss .= theme_enlight_site_news_status($theme);
    $scss .= theme_enlight_set_slide_opacity($theme);
    $scss .= theme_enlight_set_fontwww();
    $scss .= theme_enlight_set_bg_image($theme);
    if (!empty($theme->settings->scsspre)) {
        $scss .= $theme->settings->scsspre;
    }

    return $scss;
}

/**
 * Get the custom scss and add the scss into main scss.
 * @param type $theme
 * @return type
 */
function theme_enlight_get_extra_scss($theme) {

    return !empty($theme->settings->customcss) ? $theme->settings->customcss : '';
}

/**
 * Load the Jquery and migration files
 * Load the our theme js file
 *
 */
function theme_enlight_page_init(moodle_page $page) {
    global $CFG, $SESSION, $PAGE, $OUTPUT;
    $page->requires->jquery();
    $pattern = theme_enlight_get_setting('patternselect');
    $pattern = !empty($pattern) ? 'pattern-'.$pattern : "pattern-default";
    $PAGE->add_body_class($pattern);
    $PAGE->requires->js('/theme/enlight/javascript/theme.js');
    $importDemo = isset($PAGE->theme->settings->importDemo) ? $PAGE->theme->settings->importDemo : "0";
        if ( $importDemo == "1") {
            if (file_exists($CFG->dirroot.'/theme/enlight/layout/demoImporter.php')) {
                require_once($CFG->dirroot.'/theme/enlight/layout/demoImporter.php');
                $class = new enlightDemoImport;
            }
        }

    ///$page->requires->jquery_plugin('migrate');
}

/**
 * Check the frontpage site news menu status
 * @param type $theme
 * @return type
 */
function theme_enlight_site_news_status($theme) {
    $snews = $theme->settings->sitenewsType;
    if($snews == 1) {
        $scss = '$sitenewsType: true;'."\n";
        return $scss;
    } else {
        return "";
    }
}

/**
 * Set the backgrond images for the defined blocks.
 * @param type $theme
 * @return type
 */
function theme_enlight_set_bg_image($theme){
    $scss = "";
    $configurable = [
        'footbgimg_type' => 'footbgimg',
        'testimonialsbg_type' => 'testimonialsbg',
        'popularcoursesbg_type' => 'popularcoursesbg',
        "availablecoursesbg_type" => "availablecoursesbg",
        "enrolledcoursesbg_type" => "enrolledcoursesbg",
        "categoriesbg_type" => "categoriesbg",
        //'loginbg' => ['loginbg'],
    ];
     foreach ($configurable as $configkey => $target) {
        $bgType = $theme->settings->{$configkey};
        if (!empty($bgType)) {
            switch($bgType) {

                case 2:
                    $imgStatus = "true";
                    $overlayStatus = "true";
                    break;
                case 1:
                    $imgStatus = "false";
                    $overlayStatus = "true";
                    break;

                case "NONE":
                    $imgStatus = "false";
                    $overlayStatus = "false";
                    break;
                default:
                    $imgStatus = "true";
                    $overlayStatus = "true";
            }
        } else {
            $imgStatus = "true";
            $overlayStatus = "true";
        }
        $scss .= '$'.$target.'Status:'.$imgStatus. ";\n";
        $scss .= '$'.$target.'Overlay:'.$overlayStatus. ";\n";

    }
    return $scss;
}

/**
 * Get the background images.
 * @param type $theme
 * @param type $scss
 * @return type
 */
function theme_enlight_load_bgimages($theme, $scss) {

    $configurable = [
        'footbgimg'         => ['footbgimg'],
        'testimonialsbg'    => ['testimonialsbg'],
        'popularcoursesbg'  => ['popularcoursesbg'],
        'loginbg'           => ['loginbg'],
        "availablecoursesbg"=> ['availablecoursesbg'],
        "enrolledcoursesbg" => ['enrolledcoursesbg'],
        "categoriesbg"      => ['categoriesbg'],
    ];


    // Prepend variables first.
    foreach ($configurable as $configkey => $targets) {
        $value = ( $theme->setting_file_url($configkey, $configkey) ) ? $theme->setting_file_url($configkey, $configkey) : null;
        $value = theme_enlight_set_bgimg($configkey, $value);
        if (empty($value)) {
            continue;
        }
        array_map(function($target) use (&$scss, $value) {
            $scss .= '$' . $target . ': ' . $value . ";\n";
        }, (array) $targets);
    }
    return $scss;
}

/**
 * Loads the CSS Styles and replace the background images.
 * If background image not available in the settings take the default images.
 *
 * @param $css
 * @param $theme
 * @return string
 */

function theme_enlight_process_css($css, $theme) {
    global $OUTPUT, $CFG;

    $css = theme_enlight_pre_css_set_fontwww($css);




    return $css;
}



/**
 * Loads the CSS and set the background images.
 *
 * @param string $css
 * @param string $bgimage
 * @param string $setting
 * @return string
 */
function theme_enlight_set_bgimg($configkey, $value) {
    global $CFG;
    $bgimgs = array('testimonialsbg' => 'testimonialsbg.jpg', 'footbgimg' => 'footbgimg.jpg',
         'newcoursesbg' => 'newcoursesbg.jpg', 'popularcoursesbg' => 'popularcoursesbg.jpg'
        , 'aboutbg' => 'aboutbg.png', 'loginbg' => 'loginbg.jpg'
    );
    $url = "";
    if ((empty($value) || $value == null) && !empty($configkey)) {
        $url = $CFG->wwwroot.'/theme/enlight/pix/home/'.$bgimgs[$configkey];
    } else {
        $url = $value;
    }
    $imgCss = "url('".$url."')";

    return $imgCss;

}

/**
 * Get the slider obacity level from the settings and load into scss.
 * @param type $theme
 * @return type
 */
function theme_enlight_set_slide_opacity($theme) {

    if (!empty($theme->settings->slideOverlay_opacity)) {
        $opacity = $theme->settings->slideOverlay_opacity;
    } else {
        $opacity = "0.5";
    }

    $opacityCss = '$slideOverlay_opacity:' . $opacity . ";\n";


    return $opacityCss;
}

/**
 * Load the font folder path into the scss.
 * @return type
 */
function theme_enlight_set_fontwww() {
    global $CFG, $PAGE;
    if (empty($CFG->themewww)) {
        $themewww = $CFG->wwwroot."/theme";
    } else {
        $themewww = $CFG->themewww;
    }

   // $tag = '[[setting:fontwww]]';
    $theme = theme_config::load('enlight');
    $fontwww = '$fontwww: "'. $themewww.'/enlight/fonts/"'.";\n";
    //$css = str_replace($tag, $themewww.'/enlight/fonts/', $css);
    return $fontwww;
}

function theme_enlight_pre_css_set_fontwww($css) {
    global $CFG, $PAGE;
    if (empty($CFG->themewww)) {
        $themewww = $CFG->wwwroot."/theme";
    } else {
        $themewww = $CFG->themewww;
    }

    $tag = '[[setting:fontwww]]';
    $theme = theme_config::load('enlight');
    $css = str_replace($tag, $themewww.'/enlight/fonts/', $css);
    return $css;
}
/**
 * Serves any files associated with the theme settings.
 *
 * @param stdClass $course
 * @param stdClass $cm
 * @param context $context
 * @param string $filearea
 * @param array $args
 * @param bool $forcedownload
 * @param array $options
 * @return bool
 */
function theme_enlight_pluginfile($course, $cm, $context, $filearea, $args, $forcedownload, array $options = array()) {
    static $theme;
    $bgimgs = array('testimonialsbg', 'footbgimg', 'newcoursesbg', 'popularcoursesbg', 'aboutbg', 'loginbg');

    if (empty($theme)) {
        $theme = theme_config::load('enlight');
    }
    if ($context->contextlevel == CONTEXT_SYSTEM) {

        if ($filearea === 'logo') {
            return $theme->setting_file_serve('logo', $args, $forcedownload, $options);
        } else if ($filearea === 'footerlogo') {
            return $theme->setting_file_serve('footerlogo', $args, $forcedownload, $options);
        } else if ($filearea === 'style') {
            theme_enlight_serve_css($args[1]);
        } else if ($filearea === 'pagebackground') {
            return $theme->setting_file_serve('pagebackground', $args, $forcedownload, $options);
        } else if (preg_match("/slide[1-9][0-9]*image/", $filearea) !== false) {
            return $theme->setting_file_serve($filearea, $args, $forcedownload, $options);
        } else if (in_array($filearea, $bgimgs)) {
            return $theme->setting_file_serve($filearea, $args, $forcedownload, $options);
        } else {
            send_file_not_found();
        }
    } else {
        send_file_not_found();
    }
}

/**
 * Serves CSS for image file updated to styles.
 *
 * @param string $filename
 * @return string
 */
function theme_enlight_serve_css($filename) {
    global $CFG;
    if (!empty($CFG->themedir)) {
        $thestylepath = $CFG->themedir . '/enlight/style/';
    } else {
        $thestylepath = $CFG->dirroot . '/theme/enlight/style/';
    }
    $thesheet = $thestylepath . $filename;

    /* http://css-tricks.com/snippets/php/intelligent-php-cache-control/ - rather than /lib/csslib.php as it is a static file who's
      contents should only change if it is rebuilt.  But! There should be no difference with TDM on so will see for the moment if
      that decision is a factor. */

    $etagfile = md5_file($thesheet);
    // File.
    $lastmodified = filemtime($thesheet);
    // Header.
    $ifmodifiedsince = (isset($_SERVER['HTTP_IF_MODIFIED_SINCE']) ? $_SERVER['HTTP_IF_MODIFIED_SINCE'] : false);
    $etagheader = (isset($_SERVER['HTTP_IF_NONE_MATCH']) ? trim($_SERVER['HTTP_IF_NONE_MATCH']) : false);

    if ((($ifmodifiedsince) && (strtotime($ifmodifiedsince) == $lastmodified)) || $etagheader == $etagfile) {
        theme_enlight_send_unmodified($lastmodified, $etagfile);
    }
    theme_enlight_send_cached_css($thestylepath, $filename, $lastmodified, $etagfile);
}

// Set browser cache used in php header.
function theme_enlight_send_unmodified($lastmodified, $etag) {
    $lifetime = 60 * 60 * 24 * 60;
    header('HTTP/1.1 304 Not Modified');
    header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $lifetime) . ' GMT');
    header('Cache-Control: public, max-age=' . $lifetime);
    header('Content-Type: text/css; charset=utf-8');
    header('Etag: "' . $etag . '"');
    if ($lastmodified) {
        header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $lastmodified) . ' GMT');
    }
    die;
}

// Cached css.
function theme_enlight_send_cached_css($path, $filename, $lastmodified, $etag) {
    global $CFG;
    require_once($CFG->dirroot . '/lib/configonlylib.php'); // For min_enable_zlib_compression().
    // 60 days only - the revision may get incremented quite often.
    $lifetime = 60 * 60 * 24 * 60;

    header('Etag: "' . $etag . '"');
    header('Content-Disposition: inline; filename="'.$filename.'"');
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $lastmodified) . ' GMT');
    header('Expires: ' . gmdate('D, d M Y H:i:s', time() + $lifetime) . ' GMT');
    header('Pragma: ');
    header('Cache-Control: public, max-age=' . $lifetime);
    header('Accept-Ranges: none');
    header('Content-Type: text/css; charset=utf-8');
    if (!min_enable_zlib_compression()) {
        header('Content-Length: ' . filesize($path . $filename));
    }

    readfile($path . $filename);
    die;
}


/**
 * Adds any custom CSS to the CSS before it is cached.
 *
 * @param string $css The original CSS.
 * @param string $customcss The custom CSS to add.
 * @return string The CSS which now contains our custom CSS.
 */
function theme_enlight_set_customcss($css, $customcss) {
    $tag = '[[setting:customcss]]';
    $replacement = $customcss;
    if (is_null($replacement)) {
        $replacement = '';
    }

    $css = str_replace($tag, $replacement, $css);

    return $css;
}

/**
 * Returns an object containing HTML for the areas affected by settings.
 *
 * Do not add Clean specific logic in here, child themes should be able to
 * rely on that function just by declaring settings with similar names.
 *
 * @param renderer_base $output Pass in $OUTPUT.
 * @param moodle_page $page Pass in $PAGE.
 * @return stdClass An object with the following properties:
 *      - navbarclass A CSS class to use on the navbar. By default ''.
 *      - heading HTML to use for the heading. A logo if one is selected or the default heading.
 *      - footnote HTML to use as a footnote. By default ''.
 */
function theme_enlight_get_html_for_settings(renderer_base $output, moodle_page $page) {
    global $CFG;
    $return = new stdClass;

    $return->navbarclass = '';
    if (!empty($page->theme->settings->invert)) {
        $return->navbarclass .= ' navbar-inverse';
    }

    if (!empty($page->theme->settings->logo)) {
        $return->heading = html_writer::link($CFG->wwwroot, '', array('title' => get_string('home'), 'class' => 'logo'));
    } else {
        $return->heading = $output->page_heading();
    }

    $return->footnote = '';
    if (!empty($page->theme->settings->footnote)) {
        $return->footnote = '<div class="footnote text-center">'.format_text($page->theme->settings->footnote).'</div>';
    }

    return $return;
}


/**
 * Load the logo url.
 * @param type|string $type
 * @return type
 */
function theme_enlight_get_logo_url($type='header') {
    global $OUTPUT;
    static $theme;
    if (empty($theme)) {
        $theme = theme_config::load('enlight');
    }

    $logo = $theme->setting_file_url('logo', 'logo');
    $logo = empty($logo) ? $OUTPUT->image_url('home/logo', 'theme') : $logo;
    return $logo;
}
/**
 * Renderer the slider images.
 * @param type $p
 * @param type $sliname
 * @return type
 */
function theme_enlight_render_slideimg($p, $sliname) {
    global $PAGE, $OUTPUT;

    $nos = theme_enlight_get_setting('numberofslides');
    //  $i = $p % 3;
   // $slideimage = $OUTPUT->image_url('home/slide'.$i, 'theme');

    if (theme_enlight_get_setting($sliname)) {
        $slideimage = $PAGE->theme->setting_file_url($sliname, $sliname);
        return $slideimage;
    }
    return "";
}
/**
 * Description
 * @param type $setting
 * @param type|bool $format
 * @return type
 */
function theme_enlight_get_setting($setting, $format = false) {
    global $CFG;
    require_once($CFG->dirroot . '/lib/weblib.php');
    static $theme;
    if (empty($theme)) {
        $theme = theme_config::load('enlight');
    }
    if (isset($theme->settings->$setting)) {
        if (empty($theme->settings->$setting)) {
            return false;
        } else if (!$format) {
            return $theme->settings->$setting;
        } else if ($format === 'format_text') {
            return format_text($theme->settings->$setting, FORMAT_PLAIN);
        } else if ($format === 'format_html') {
            return format_text($theme->settings->$setting, FORMAT_HTML, array('trusted' => true, 'noclean' => true));
        } else {
            return format_string($theme->settings->$setting);
        }
    }
}

/**
 * Render the current theme url
 *
 * @param void
 * @return string
 */
function theme_enlight_theme_url() {
    global $CFG, $PAGE;
    $themeurl = $CFG->wwwroot.'/theme/'. $PAGE->theme->name;
    return $themeurl;
}

/**
 * Display Footer Block Custom Links
 * @param string $menu_name Footer block link name.
 * @return string The Footer links are return.
 */

function theme_enlight_generate_links($menuname = '') {
    global $CFG, $PAGE;
    $htmlstr = '';
    $menustr = theme_enlight_get_setting($menuname);
    $menusettings = explode("\n", $menustr);
    foreach ($menusettings as $menukey => $menuval) {
        $expset = explode("|", $menuval);
        if (!empty($expset) && isset($expset[0]) && isset($expset[1])) {
            list($ltxt, $lurl) = $expset;
            $ltxt = trim($ltxt);
            $ltxt = theme_enlight_lang($ltxt);
            $lurl = trim($lurl);
            if (empty($ltxt)) {
                continue;
            }
            if (empty($lurl)) {
                $lurl = 'javascript:void(0);';
            }
            $pos = strpos($lurl, 'http');
            if ($pos === false) {
                $lurl = new moodle_url($lurl);
            }
            $htmlstr .= '<li><a href="'.$lurl.'">'.$ltxt.'</a></li>'."\n";
        }
    }
    return $htmlstr;
}

/**
 * Display Footer block Social Media links.
 *
 * @return string The Footer Social Media links are return.
 */
function theme_enlight_social_links() {
    global $CFG;
    $totalicons = 4;
    $htmlstr = '';
    for ($i = 1; $i <= 4; $i++) {
        $iconenable = theme_enlight_get_setting('siconenable'.$i);
        $icon = theme_enlight_get_setting('socialicon'.$i);
        $iconcolor = theme_enlight_get_setting('siconbgc'.$i);
        $iconurl = theme_enlight_get_setting('siconurl'.$i);
        $iconstr = '';
        $iconsty = (empty($iconcolor)) ? '' : ' style="background: '.$iconcolor.';"';
        if ($iconenable == "1") {
            $iconstr = '<li class="media0'.$i.'"'.$iconsty.'><a href="'.$iconurl.'"><i class="fa fa-'.$icon.'"></i></a></li>'."\n";
            $htmlstr .= $iconstr;
        }
    }
    return $htmlstr;
}

/**
 * Remove the html special tags from course content.
 * This function used in course home page.
 *
 * @param string $text
 * @return string
 */
function theme_enlight_strip_html_tags( $text ) {
    $text = preg_replace(
        array(
            // Remove invisible content.
            '@<head[^>]*?>.*?</head>@siu',
            '@<style[^>]*?>.*?</style>@siu',
            '@<script[^>]*?.*?</script>@siu',
            '@<object[^>]*?.*?</object>@siu',
            '@<embed[^>]*?.*?</embed>@siu',
            '@<applet[^>]*?.*?</applet>@siu',
            '@<noframes[^>]*?.*?</noframes>@siu',
            '@<noscript[^>]*?.*?</noscript>@siu',
            '@<noembed[^>]*?.*?</noembed>@siu',
            // Add line breaks before and after blocks.
            '@</?((address)|(blockquote)|(center)|(del))@iu',
            '@</?((div)|(h[1-9])|(ins)|(isindex)|(p)|(pre))@iu',
            '@</?((dir)|(dl)|(dt)|(dd)|(li)|(menu)|(ol)|(ul))@iu',
            '@</?((table)|(th)|(td)|(caption))@iu',
            '@</?((form)|(button)|(fieldset)|(legend)|(input))@iu',
            '@</?((label)|(select)|(optgroup)|(option)|(textarea))@iu',
            '@</?((frameset)|(frame)|(iframe))@iu',
        ),
        array(
            ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ',
            "\n\$0", "\n\$0", "\n\$0", "\n\$0", "\n\$0", "\n\$0",
            "\n\$0", "\n\$0",
        ),
        $text
    );
    return strip_tags( $text );
}

/**
 * Cut the Course content.
 *
 * @param $str
 * @param $n
 * @param $end_char
 * @return string
 */
function theme_enlight_course_trim_char($str, $n = 500, $endchar = '&#8230;') {
    if (strlen($str) < $n) {
        return $str;
    }

    $str = preg_replace("/\s+/", ' ', str_replace(array("\r\n", "\r", "\n"), ' ', $str));
    if (strlen($str) <= $n) {
        return $str;
    }

    $out = "";
    $small = substr($str, 0, $n);
    $out = $small.$endchar;
    return $out;
}

function theme_enlight_tmonials() {
    global $OUTPUT, $PAGE;

    $toggletmonial = theme_enlight_get_setting('toggletmonial');
    if (!$toggletmonial) {
        return '';
    }
    $strtmonial = '';
    $numberoftmonials = theme_enlight_get_setting('numberoftmonials');
    $traversalStatus = theme_enlight_get_setting('tmonialTraversal_status');
    $atestimonials = array();
    $nouserimg = $OUTPUT->image_url('no-user', 'theme');
    $titems = '';
    $indicators = '';
    $traverser = 0;
    for ($i = 1; $i <= $numberoftmonials; $i++) {
        $testimonial = theme_enlight_get_setting('tmonial'.$i.'text', 'format_text');
        $uname = theme_enlight_get_setting('tmonial'.$i.'uname');
        $uimg = $PAGE->theme->setting_file_url('tmonial'.$i.'img', 'tmonial'.$i.'img');
        $uimg = empty($uimg) ? $nouserimg : $uimg;

        if (!empty($testimonial) && !empty($uname) ) {
            $clstxt = ($i == 1) ? 'active ' : '';
            $titems .= html_writer::start_tag('div', array('class'=>'content ' ));
            $titems .= html_writer::start_tag('div', array('class'=>'item-content ' ));
            $titems .= html_writer::tag('p',$testimonial);
            $titems .= html_writer::start_tag('div', array('class' => 'user-info'));
            $titems .= html_writer::start_tag('div', array('class' => 'thumb'));
            $titems .= html_writer::empty_tag('img',array('src'=>$uimg, 'width' => '128', 'height'=> '128', 'alt' => $uname));

            $titems .= html_writer::end_tag('div');
            $titems .= html_writer::tag('h6', $uname);
            $titems .= html_writer::end_tag('div');
            $titems .= html_writer::end_tag('div');
            $titems .= html_writer::end_tag('div');
        }
    }

    $strtestimonial = html_writer::start_tag('div', array('class'=>'frontpage-testimonials ' ));
    $strtestimonial .= html_writer::tag('div','', array('class'=>'bgtrans-overlay'));
    $strtestimonial .= html_writer::start_tag('div', array('class'=>'container ' ));
    $strtestimonial .= html_writer::start_tag('div', array('class'=>'row' ));
    $strtestimonial .= html_writer::start_tag('div', array('class'=>'col-md-12' ));
    $strtestimonial .= html_writer::start_tag('div', array('class'=>'slide', 'id'=>'Carouseltestimonials' ));

    $strtestimonial .= $titems;
    $strtestimonial .= html_writer::end_tag('div');
    $strtestimonial .= html_writer::end_tag('div');
    $strtestimonial .= html_writer::end_tag('div');
    $strtestimonial .= html_writer::end_tag('div');
    $strtestimonial .= html_writer::end_tag('div');
    // $strtestimonial .= html_writer::end_tag('div');


    return $strtestimonial;
}

function theme_enlight_marketingspot1() {
    global $CFG, $PAGE;

    $status = theme_enlight_get_setting('marketingSpot1_status');
    $description = theme_enlight_get_setting('mspot1desc');
    $title = theme_enlight_lang(theme_enlight_get_setting('mspot1title'));
    $media = theme_enlight_get_setting('mspot1media','format_html');
    if (!empty($media)) {
        $class_media = 'video-visible';
    } else{
        $class_media = "";
    }

    $content = '';
    if (!empty($title)) {
        $title = explode(' ', $title);
        $title1 = (array_key_exists(0, $title)) ? $title[0] : $title;
        $title2 = array_slice($title, 1);
        $title2 = implode(' ', $title2);
    } else{

        $title1 = $title2 = "";
    }
    if (isset($status) && $status == 1 ) {
        $description = theme_enlight_lang($description);
        if (!empty($description) || !empty($media)) {

            if (!empty($media) && !empty($description) ) {
                $hide = "display:none";
                $hide2 = "";
            } else if (!empty($media) && empty($description)) {
                $hide = "";
                $hide2 = "display:none";
            } else {
                $hide = "";
                $hide2 = "display:none";
            }
            $content .= html_writer::start_tag('div', array('class' => 'site-info' ));
                $content .= html_writer::start_tag('div', array('class' => 'container'));

                    $content .= html_writer::start_tag('div', array('class' => 'info-content '. $class_media ));
                        $content .= html_writer::tag('h2', html_writer::tag('b', $title1) ." ".$title2, array('style' => $hide) );

                        $content .= html_writer::start_tag('div', array('class' => 'info-video'));
                        $content .= $media;
                        $content .= html_writer::end_tag('div');

                        if (!empty($description)) {
                            $content .= html_writer::start_tag('div', array('class' => 'info-block'));
                                $content .= html_writer::tag('h2', html_writer::tag('b', $title1) ." ".$title2, array('style' => $hide2) );

                                $content .= html_writer::tag('p', $description);
                            $content .= html_writer::end_tag('div');
                        }
                    $content .= html_writer::end_tag('div');
                $content .= html_writer::end_tag('div');
            $content .= html_writer::end_tag('div');
        }

    }
    return $content;
}

function theme_enlight_category_menu() {
    global $CFG, $PAGE;
    $categoryid = optional_param('categoryid', null, PARAM_INT);
    $category = coursecat::get($categoryid);
    $html = '';
    if ($category === null) {
        $selectedparents = array();
        $selectedcategory = null;
    } else {
        $selectedparents = $category->get_parents();
        $selectedparents[] = $category->id;
        $selectedcategory = $category->id;
    }

    $catatlevel = \core_course\management\helper::get_expanded_categories('');
    $catatlevel[] = array_shift($selectedparents);
    $catatlevel = array_unique($catatlevel);

    require_once($CFG->libdir. '/coursecatlib.php');
    $listing = coursecat::get(0)->get_children();
    $html .= '<ul class="nav">';
    foreach ($listing as $listitem) {
        $subcategories = array();
        if (in_array($listitem->id, $catatlevel)) {
            $subcategories = $listitem->get_children();
        }
        $html .= theme_enlight_category_menu_item(
        $listitem,
        $subcategories,
        $listitem->get_children_count(),
        $selectedcategory,
        $selectedparents
        );
    }
    $html .= '</ul>';
    return $html;
}

function theme_enlight_category_menu_item(coursecat $category, array $subcategories, $totalsubcategories,
$selectedcategory = null, $selectedcategories = array()) {

    $viewcaturl = new moodle_url('/course/index.php', array('categoryid' => $category->id));
    $text = $category->get_formatted_name();
    $isexpandable = ($totalsubcategories > 0);
    $isexpanded = (!empty($subcategories));
    $activecategory = ($selectedcategory === $category->id);
    $dataexpanded = $isexpanded ? ' data-expanded = "1" ' : ' data-expanded = "0"';

    if ($isexpanded) {
        $cls = $activecategory ? 'has-children expanded' : 'has-children';
    } else if ($isexpandable) {
        $cls = 'has-children';
    } else {
        $cls = $activecategory ? 'expanded' : '';
    }

    $html = '<li class="'.$cls.'"'.$dataexpanded.'>';
    $html .= '<a href="'.$viewcaturl.'">'.$text.'</a>';

    if (!empty($subcategories)) {
        $html .= '<ul class="nav childnav">';

        $catatlevel = \core_course\management\helper::get_expanded_categories($category->path);
        $catatlevel[] = array_shift($selectedcategories);
        $catatlevel = array_unique($catatlevel);

        foreach ($subcategories as $listitem) {
            $childcategories = (in_array($listitem->id, $catatlevel)) ? $listitem->get_children() : array();
            $html .= theme_enlight_category_menu_item(
            $listitem,
            $childcategories,
            $listitem->get_children_count(),
            $selectedcategory,
            $selectedcategories
            );
        }

        $html .= '</ul>';
    }
    $html .= '</li>';

    return $html;
}

function theme_enlight_get_courses_page1($categoryid="all", $sort="c.sortorder ASC", $fields="c.*",
&$totalcount, $limitfrom="", $limitnum="") {

    global $USER, $CFG, $DB;
    $params = array();

    $categoryselect = "";
    if ($categoryid !== "all" && is_numeric($categoryid)) {
        $categoryselect = "WHERE c.category = :catid and c.visible = :visible and c.id != '1' ";
        $params['catid'] = $categoryid;
        $params['visible'] = '1';
    } else {
        $categoryselect = "WHERE  c.visible = :visible and c.id != '1' ";
        $params['visible'] = '1';
    }

    $ccselect = ', ' . context_helper::get_preload_record_columns_sql('ctx');
    $ccjoin = "LEFT JOIN {context} ctx ON (ctx.instanceid = c.id AND ctx.contextlevel = :contextlevel)";
    $params['contextlevel'] = CONTEXT_COURSE;

    $totalcount = 0;
    if (!$limitfrom) {
        $limitfrom = 0;
    }
    $visiblecourses = array();

    $sql = "SELECT $fields $ccselect
        FROM {course} c
        $ccjoin
        $categoryselect
        ORDER BY $sort";

    // Pull out all course matching the cat.
    $rs = $DB->get_recordset_sql($sql, $params);
    // Iteration will have to be done inside loop to keep track of the limitfrom and limitnum.
    foreach ($rs as $course) {
        context_helper::preload_from_record($course);
        if ($course->visible <= 0) {
            // For hidden courses, require visibility check.
            if (has_capability('moodle/course:viewhiddencourses', context_course::instance($course->id))) {
                $totalcount++;
                if ($totalcount > $limitfrom && (!$limitnum or count($visiblecourses) < $limitnum)) {
                    $visiblecourses [$course->id] = $course;
                }
            }
        } else {
            $totalcount++;
            if ($totalcount > $limitfrom && (!$limitnum or count($visiblecourses) < $limitnum)) {
                $visiblecourses [$course->id] = $course;
            }
        }
    }
    $rs->close();
    return $visiblecourses;

}

function theme_enlight_lang($key='') {
    $pos = strpos($key, 'lang:');
    if ($pos !== false) {
        list($l, $k) = explode(":", $key);
        if (get_string_manager()->string_exists($k, 'theme_enlight')) {
            $v = get_string($k, 'theme_enlight');
            return $v;
        } else {
            return $key;
        }
    } else {
        return $key;
    }

}
/**
 * Check the memeber status for show the person.
 * @param type $nums
 * @return type
 */
function theme_enlight_check_our_team_status($nums) {
    foreach ($nums as $key => $value) {
        $con = theme_enlight_get_team_user($value, 'check');
        if ($con == "") {
            unset($nums[$key]);
        }
    }
    return $nums;
}

/**
 * This code returns the our team block contents.
 * @return type
 */
function theme_enlight_our_team() {
    global $OUTPUT;
    $numberofpersons = theme_enlight_get_setting('numberofpersons');
    $num = range(1, $numberofpersons);
    $checkstatus = theme_enlight_check_our_team_status($num);
    if (!empty($checkstatus)) {
        $rows = array_chunk($checkstatus, 4);
    } else {
       return "";
    }
    $title = theme_enlight_get_setting('ourteamTitle');
    $title = theme_enlight_lang($title);

    ///$rows = array_chunk($numberofpersons, 4);

    $teamContent = "";

    $teamContent .= html_writer::start_tag('div', array('class' => 'our-team'));
    $teamContent .= html_writer::start_tag('div', array('class' => 'container'));
    $teamContent .= html_writer::tag('h2', $title);
    if (count($checkstatus) >= 5) {
        $teamContent .= '<div class="pagenav slider-nav">
                        <button class="slick-prev nav-item previous" type="button">
                        <button class="slick-next nav-item next" type="button">
                        </button>
                        <div class="clearfix"></div>
                   </div>
                   <div class="clearfix"></div>';
    }
    $teamContent .= html_writer::start_tag('div', array('class' => 'our_team_slider', 'data-crow' => '4'));
    foreach ($rows as $row ) {
        $teamContent .= html_writer::start_tag('div',array('class'=> 'row '));
        foreach($row as $i) {

            $value = theme_enlight_get_team_user($i);
            if (!empty($value)) {
                $img = $OUTPUT->image_url("home/img",'theme');
                $nouserimg = $OUTPUT->image_url('no-user', 'theme');
                $teamContent .= html_writer::start_tag('div', array('class'=> 'col-md-3'));
                $teamContent .= html_writer::start_tag('div', array('class' => 'card-block '));
                $teamContent .= html_writer::start_tag('div', array('class' => 'card-img'));
                $teamContent .= html_writer::empty_tag('img', array('src' => $value['userImg'] , 'width' => '150', 'height' => '150', 'alt' => 'our-team-1')); // Person Image.

                $teamContent .= html_writer::start_tag('div', array('class' => 'social-link'));


                if (!empty($value['userSocial_icon1'])) {
                    $teamContent .= html_writer::link($value['userSocial_url1'], html_writer::tag('i', '', array('class' => 'fa '.$value['userSocial_icon1'])), array('class'=>'icon-link') );
                }
                if (!empty($value['userSocial_icon2'])) {
                    $teamContent .= html_writer::link($value['userSocial_url2'], html_writer::tag('i', '', array('class' => 'fa '.$value['userSocial_icon2'])), array('class' => 'icon-link') );
                }
                if (!empty($value['userSocial_icon3'])) {
                    $teamContent .= html_writer::link($value['userSocial_url3'], html_writer::tag('i', '', array('class' => 'fa '.$value['userSocial_icon3'])), array('class' => 'icon-link') );
                }
                if (!empty($value['userSocial_icon4'])) {
                    $teamContent .= html_writer::link($value['userSocial_url4'], html_writer::tag('i', '', array('class' => 'fa '.$value['userSocial_icon4'])), array('class' => 'icon-link') );
                }
                $teamDescription_length = 500;
                $description = check_length($value['description'], $teamDescription_length);
                $description = (isset($description) && !empty($description)) ? $description : "";
                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of social-link -->.

                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of Card-img -->.
                $teamContent .= html_writer::start_tag('div', array('class' => 'card-content'));
                $teamContent .= html_writer::start_tag('div', array('class' => 'card-header'));
                $teamContent .= html_writer::tag('h6', $value['username'] ); // Our Team Person Name.
                $teamContent .= html_writer::tag('span',$value['userPosition']);
                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of Card-header -->.
                $teamContent .= html_writer::start_tag('div', array('class' => 'card-element'));
                $teamContent .= html_writer::tag('p', $description); // Team person description.
                if (!empty($value['userProfileurl'])) {
                    $teamContent .= html_writer::start_tag('div', array('class' => 'card-link'));
                    $teamContent .= html_writer::link($value['userProfileurl'], get_string('seeprofile', 'theme_enlight'));
                    $teamContent .= html_writer::end_tag('div'); // Div <!-- End of card-link -->.
                }
                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of card-element -->.
                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of card-content -->.
                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of card-block -->.
                $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of card-block -->.
            }
        }
   // if ($numberofpersons % 4 != 0) {
        $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of container -->.
    }
    $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of container -->.
    $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of container -->.
    $teamContent .= html_writer::end_tag('div'); // Div <!-- End Of Our Team -->.

    return $teamContent;
}

function theme_enlight_get_team_user($i, $type = "check") {

global $CFG, $PAGE, $OUTPUT;
$username = theme_enlight_get_setting('team'.$i.'pname');
$userImg = $PAGE->theme->setting_file_url('team'.$i.'img', 'team'.$i.'img');
$userPosition = theme_enlight_get_setting('team'.$i.'position');
$userProfileurl = theme_enlight_get_setting('team'.$i.'profileUrl');

$userSocial_icon1 = theme_enlight_get_setting('team'.$i.'socialIcon1');
$userSocial_icon2 = theme_enlight_get_setting('team'.$i.'socialIcon2');
$userSocial_icon3 = theme_enlight_get_setting('team'.$i.'socialIcon3');
$userSocial_icon4 = theme_enlight_get_setting('team'.$i.'socialIcon4');

$userSocial_url1 = theme_enlight_get_setting('team'.$i.'sIconurl1');
$userSocial_url2 = theme_enlight_get_setting('team'.$i.'sIconurl2');
$userSocial_url3 = theme_enlight_get_setting('team'.$i.'sIconurl3');
$userSocial_url4 = theme_enlight_get_setting('team'.$i.'sIconurl4');

$description = theme_enlight_get_setting('team'.$i.'description');
 $nouserimg = $OUTPUT->image_url('no-user', 'theme');

$content = array();
$content['username'] = (isset($username)) ? $username : "";
$content['userPosition'] = (isset($userPosition)) ? $userPosition : "";
$content['userProfileurl'] = (isset($userProfileurl)) ? $userProfileurl : "";
$content['userSocial_icon1'] = (isset($userSocial_icon1)) ? trim($userSocial_icon1) : "";
$content['userSocial_icon2'] = (isset($userSocial_icon2)) ? trim($userSocial_icon2) : "";
$content['userSocial_icon3'] = (isset($userSocial_icon3)) ? trim($userSocial_icon3) : "";
$content['userSocial_icon4'] = (isset($userSocial_icon4)) ? trim($userSocial_icon4) : "";

$content['userSocial_url1'] = (!empty($userSocial_url1)) ? $userSocial_url1 : "#";
$content['userSocial_url2'] = (!empty($userSocial_url2)) ? $userSocial_url2 : "#";
$content['userSocial_url3'] = (!empty($userSocial_url3)) ? $userSocial_url3 : "#";
$content['userSocial_url4'] = (!empty($userSocial_url4)) ? $userSocial_url4 : "#";

$content['description'] = (isset($description)) ? $description : "";
$content['userImg'] = (!empty($userImg)) ? $userImg : "";
//if ($type == "check") {
    if (empty($content['username']) && empty($content['userPosition']) && empty($content['description']) && empty($content['userImg']) ) {
        return false;
    }
    $content['userImg'] = (!empty($userImg)) ? $userImg : $nouserimg;
//}
return $content;
}

function check_length($value, $length) {

    if(strlen($value) <= $length) {
        return $value;
    }
    else {
        $content = substr($value, 0, $length) . '...';
        return $content;
    }
}
function theme_enlight_combolist_type(){
    global $PAGE;
    $type = theme_enlight_get_setting('comboListboxType');
    if ($type == 1) {
        return true;
    } else {
        return false;
    }
}



function theme_enlight_footer_address($check = "") {
    global $PAGE;
    $value = '';
    $address = theme_enlight_get_setting('footaddress');
    $address = theme_enlight_lang($address);
    $email = theme_enlight_get_setting('footemailid');
    $email = theme_enlight_lang($email);
    $phone = theme_enlight_get_setting('footphoneno');
    $phone = theme_enlight_lang($phone);
    if (!empty($address) || !empty($email) || !empty($phone)) {
        $status = "true";
        $value = html_writer::start_tag('div', array('class' => 'footer-address-block'));

        if (!empty($address)) {
            $value .= html_writer::start_tag('div', array('class' => 'footer-address'));
            $value .= html_writer::tag('p', "<i class='fa fa-map-marker'></i>".$address);
            $value .= html_writer::end_tag('div');
        }
        if (!empty($phone)) {
            $value .= html_writer::start_tag('div', array('class' => 'footer-phone'));
            $value .= html_writer::start_tag('p');
            $value .= "<i class='fa fa-phone-square'></i>".get_string('phone').": ";
            $value .= $phone;
            $value .= html_writer::end_tag('p');
            $value .= html_writer::end_tag('div');
        }
        if (!empty($email)) {
            $value .= html_writer::start_tag('div', array('class' => 'footer-email'));
            $value .= html_writer::start_tag('p');

            $value .= "<i class='fa fa-envelope'></i>".get_string('emailid', 'theme_enlight').": ";
            $value .= html_writer::link($email, $email);
            $value .= html_writer::end_tag('p');
            $value .= html_writer::end_tag('div');
        }
        $value .= html_writer::end_tag('div');
    } else {
        $status = "false";
        $value = "";
    }
    if ($check == "true") {
        return $status;
    }
    return $value;
}

function get_theme_loader() {
    global $OUTPUT, $PAGE;
    $loaderStatus = theme_enlight_get_setting('preloader_status');
    if ($loaderStatus == '1') {
        $loadercolor = theme_enlight_get_setting('preloaderbg_color');
        $loadercolor =(isset($loadercolor)&& !empty($loadercolor)) ? $loadercolor : "#353535";
        $loaderlogo = theme_enlight_get_logo_url();
       /* $loaderlogo = (isset($loaderlogo) && !empty($loaderlogo)) ? $loaderlogo : $OUTPUT->image_url("home/logo", 'theme');*/

        $loaderimg = $PAGE->theme->setting_file_url('preloader', 'preloader');
        $loaderimg = (!empty($loaderimg)) ? $loaderimg : $OUTPUT->image_url('home/loader', 'theme');
        $loader = html_writer::start_tag('div', array('class'=> 'theme_loader', 'id' => 'theme_loader', 'style' => 'background-color: '.$loadercolor.';'));

        $loader .= html_writer::empty_tag('img', array('src' => $loaderlogo, 'class' => '' ))."<br>";
        if (!empty($loader)) {
            $loader .= html_writer::empty_tag('img', array('src' => $loaderimg, 'class' => 'loader-img' ));
        }
        $loader .= html_writer::end_tag('div');
    } else {
        $loader = "";
    }

    return $loader;
}

function theme_enlight_header_style() {
    global $CFG, $PAGE;
    $headerStyle = theme_enlight_get_setting('headerStyle');
    $headerStyle = (!empty($headerStyle)) ? $headerStyle : "1";
    if ($headerStyle == '2') {
        return '2';
    } else{
        return  '';
    }
}


